package furny.swing.common;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.io.File;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.filechooser.FileFilter;

/**
 * Panel to choose files.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class FileChooserPanel extends JPanel {
  private JTextField textField;
  private File lastPath = new File(".");
  private File file;
  private JButton button;
  private FileChosenListener listener;
  private FileFilter filter;

  private int fileSelectionMode = JFileChooser.FILES_AND_DIRECTORIES;

  /**
   * Instantiates a new file chooser panel.
   * 
   * @param listener
   *          the listener
   * @param filter
   *          the filter
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public FileChooserPanel(final FileChosenListener listener,
      final FileFilter filter) {
    this.listener = listener;
    this.filter = filter;

    setLayout(new BorderLayout());

    textField = new JTextField(" ", 10);
    setBorder(textField.getBorder());
    textField.setBorder(BorderFactory.createEmptyBorder());
    textField.addFocusListener(new FocusAdapter() {
      @Override
      public void focusLost(final FocusEvent e) {
        if (file != null) {
          textField.setText(file.getAbsolutePath());
          textField.setToolTipText(file.getAbsolutePath());
        } else {
          textField.setText("");
          textField.setToolTipText("");
        }
      }
    });

    add(textField, BorderLayout.CENTER);

    setBackground(textField.getBackground());

    button = new JButton(new ActionFileBrowse()) {
      @Override
      public Dimension getPreferredSize() {

        return new Dimension(18, 18);
      }

      @Override
      public void setSize(final Dimension d) {
        super.setSize(getPreferredSize());
      }
    };
    button.setMargin(new Insets(3, 3, 3, 3));

    button.setOpaque(false);
    button.setBorder(BorderFactory.createEmptyBorder());

    add(button, BorderLayout.EAST);
  }

  /**
   * Gets the selected file.
   * 
   * @return the selected file
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public File getSelectedFile() {
    return file;
  }

  /**
   * Sets a file.
   * 
   * @param file
   *          the new file
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFile(final File file) {
    this.file = file;
    final String name = file != null ? file.getAbsolutePath() : "";
    textField.setText(name);
    final String path = file != null ? file.getAbsolutePath() : "";
    textField.setToolTipText(path);
    if (file != null) {
      if (file.isDirectory()) {
        lastPath = file;
      } else {
        lastPath = file.getParentFile();
      }
    }

    listener.fileChosen(file, 0);
  }

  /**
   * Sets the file selection mode.
   * 
   * @param fileSelectionMode
   *          the new file selection mode
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFileSelectionMode(final int fileSelectionMode) {
    this.fileSelectionMode = fileSelectionMode;
  }

  /**
   * Sets a file by its name.
   * 
   * @param fileName
   *          the new file name
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFile(final String fileName) {
    File f = null;
    if (fileName != null) {
      f = new File(fileName);
    }
    setFile(f);
  }

  /**
   * Action to start the file browser.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionFileBrowse extends AbstractAction {

    /**
     * Instantiates a new action for file browsing.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionFileBrowse() {
      super("...");
      putValue(SHORT_DESCRIPTION, "Browse for a file");
    }

    @Override
    public void actionPerformed(final ActionEvent arg0) {
      final JFileChooser chooser = new JFileChooser(lastPath);
      chooser.setFileFilter(filter);
      chooser.setFileSelectionMode(fileSelectionMode);
      final int returnVal = chooser.showOpenDialog(getParent());
      if (returnVal == JFileChooser.APPROVE_OPTION) {
        file = chooser.getSelectedFile();
        setFile(file);
      }
    }
  }
}
